clear all
capture log close
log using ddwc_rep_fars_extract, replace text
set more off

/* SPECIFY THE FARS WAVES OVER WHICH TO AGGREGATE (GENERALLY, ALL AVAILABLE YEARS). */
local firstyr = 1982
*local firstyr = 2014
local lastyr = 2017

/* LOOP OVER ALL OF THE ANNUAL ACCIDENT, PERSON, VEHICLE, AND MIPER (MULTIPLE IMPUTATION) FILES, TO COMBINE INTO ONE DATASET */
/* 	IN THE 2009, 2011, 2012 DTA FILES, THERE WERE ABOUT 300 TOTAL ACCIDENTS MISSING IN EACH YEAR COMPARED WITH THE NEWER CSV FILES */

forvalues yr = `firstyr'/`lastyr' {
*local yr = 2017

	* PREPARE THE ANNUAL ACCIDENT VARIABLES
	import delimited "FARS`yr'/accident", clear
	replace year = year + 1900 if `yr' <= 1997
	mvdecode hour, mv(99)
	recode hour (24 = 0)
	mvdecode day_week, mv(9)
	recode month (1 2 3 = 1) (4 5 6 = 2) (7 8 9 = 3) (10 11 12 = 4), gen(quarter)	
	/* STATE IDENTIFIERS */
	rename state fips
	* file that assigns state abbreviations based on state FIPS code
	merge m:1 fips using "statecodes.dta", keep(master match) keepusing(abbr) nogen
	rename (fips abbr) (state state_abbr)
	
	keep year st_case state state_abbr quarter day_week hour persons
	if `yr' > `firstyr' capture append using ddwc_accident
	save ddwc_accident, replace

	* PREPARE THE ANNUAL VEHICLE VARIABLES
	import delimited "FARS`yr'/vehicle", clear
	gen year = `yr'
	
	/* NUMBER OF OCCUPANTS */
	if `yr' <= 2008 gen occupants = ocupants
	else gen occupants = numoccs
	if `yr' <= 2015 replace occupants = . if occupants >= 99
	else replace occupants = . if occupants >= 97
	
	/* VEHICLE MODEL YEAR */
	mvdecode mod_year, mv(0 99 9998 9999)
	replace mod_year = mod_year + 1900 if `yr' <= 1997
	
	/* DRIVER PREVIOUS VIOLATIONS */
	mvdecode prev_*, mv(98 99 998)
	
	keep year st_case veh_no prev_acc prev_sus prev_dwi prev_spd prev_oth mod_year dr_drink occupants
	if `yr' > `firstyr' capture append using ddwc_vehicle
	save ddwc_vehicle, replace
	
	* PREPARE THE ANNUAL PERSON VARIABLES
	
	/* FIRST PREPARE THE MI FILE FOR MERGING */
	tempfile tmp_miper`yr'
	import delimited "FARS`yr'/miper", clear
	save `tmp_miper`yr''
	
	/* IMPORT PERSON FILE */
	import delimited "FARS`yr'/person", clear
	gen year = `yr'
	
	/* ALCOHOL TEST RESULT */
	if `yr' <= 1990 gen alcohol_test_result = test_res
	else gen alcohol_test_result = alc_res
	if `yr' >= 2015 replace alcohol_test_result = alcohol_test_result/10
	replace alcohol_test_result = . if alcohol_test_result >= 95
	
	/* AGE OF DRIVER OR PASSENGER */
	gen age_lt15 = age < 15
	
	/* DEMOGRAPHIC CHARACTERISTICS: NOTE THAT RACE IS NOT INCLUDED HERE BECAUSE RACE IS ONLY RECORDED FOR PEOPLE WHO DIED */
	gen dr_male = (sex == 1) if seat_pos == 11 & sex < 9
	gen dr_age = age if seat_pos == 11 & age < 998
	
	/* DRIVER MULTIPLE IMPUTED BAC LEVELS */
	merge 1:1 st_case veh_no per_no using `tmp_miper`yr'', keep(master match) nogen
	
	keep year st_case veh_no per_no seat_pos drinking alcohol_test_result age_lt15 dr_age dr_male p1-p10
	if `yr' > `firstyr' capture append using ddwc_person
	save ddwc_person, replace
}

*if `yr' > `firstyr' append using ddwc_fars_extract
*save ddwc_rep_fars_extract, replace

log close 

